<?php
/**
 * Plugin Name: Easy Abandoned Cart
 * Description: Sends up to 3 follow-up emails to WooCommerce customers who abandon their cart. Includes admin panel to customize templates, set sender address, and send test emails.
 * Version: 2.3
 * Author: Anna Nicole Kucerova
 */

defined('ABSPATH') || exit;

// === 1. Schedule abandoned cart follow-ups ===
add_action('woocommerce_order_status_pending', function($order_id) {
    wp_schedule_single_event(time() + 3600, 'eac_send_email_1', [$order_id]);     // after 1 hour
    wp_schedule_single_event(time() + 86400, 'eac_send_email_2', [$order_id]);    // after 1 day
    wp_schedule_single_event(time() + 259200, 'eac_send_email_3', [$order_id]);   // after 3 days
});

// === 2. General email send handler ===
function eac_process_email($order_id, $index) {
    $order = wc_get_order($order_id);
    if (!$order || $order->get_status() !== 'pending') return;

    $to         = $order->get_billing_email();
    $first_name = $order->get_billing_first_name();
    $shop_name  = get_bloginfo('name');
    $contact    = get_option('woocommerce_email_from_address');
    $order_url  = $order->get_checkout_payment_url();

    $from_email = get_option('eac_sender_email', $contact);
    $from_name  = get_option('eac_sender_name', $shop_name);

    $template = get_option("eac_email_template_$index") ?: "<h2>Hello {{first_name}},</h2><p>You left something in your cart.</p><p><a href='{{order_url}}'>Complete your order</a></p><p>Thanks,<br>{{shop_name}}</p>";

    $message = str_replace(
        ['{{first_name}}', '{{shop_name}}', '{{contact_email}}', '{{order_url}}'],
        [$first_name, $shop_name, $contact, esc_url($order_url)],
        $template
    );

    $headers = [
        'Content-Type: text/html; charset=UTF-8',
        'From: ' . $from_name . ' <' . $from_email . '>',
    ];

    wp_mail($to, "Did you forget something at $shop_name?", $message, $headers);
}

add_action('eac_send_email_1', fn($order_id) => eac_process_email($order_id, 1));
add_action('eac_send_email_2', fn($order_id) => eac_process_email($order_id, 2));
add_action('eac_send_email_3', fn($order_id) => eac_process_email($order_id, 3));

// === 3. Admin menu ===
add_action('admin_menu', function() {
    add_menu_page('Easy Abandoned Cart', 'EasyAbandonedCart', 'manage_options', 'easy-abandoned-cart', 'eac_admin_page', 'dashicons-email', 60);
});

// === 4. Admin page UI ===
function eac_admin_page() {
    echo '<div class="wrap"><h1>Easy Abandoned Cart – Email Settings</h1>';

    // Save sender settings
    if (isset($_POST['eac_save_sender'])) {
        update_option('eac_sender_email', sanitize_email($_POST['eac_sender_email']));
        update_option('eac_sender_name', sanitize_text_field($_POST['eac_sender_name']));
        echo '<div class="updated"><p>Sender settings saved.</p></div>';
    }

    // Save templates
    for ($i = 1; $i <= 3; $i++) {
        if (isset($_POST["eac_save_template_$i"])) {
            update_option("eac_email_template_$i", wp_kses_post($_POST["eac_email_template_$i"]));
            echo "<div class='updated'><p>Email template #$i saved.</p></div>";
        }
    }

    // Send test email
    if (isset($_POST['eac_send_test_email']) && isset($_POST['eac_test_email'])) {
        $test_to     = sanitize_email($_POST['eac_test_email']);
        $template_id = intval($_POST['eac_test_template']);

        $template = get_option("eac_email_template_$template_id") ?: "<p>This is a test email.</p>";
        $shop_name  = get_bloginfo('name');
        $sender     = get_option('woocommerce_email_from_address');
        $from_email = get_option('eac_sender_email', $sender);
        $from_name  = get_option('eac_sender_name', $shop_name);

        $message = str_replace(
            ['{{first_name}}', '{{shop_name}}', '{{contact_email}}', '{{order_url}}'],
            ['TestUser', $shop_name, $sender, home_url()],
            $template
        );

        $headers = [
            'Content-Type: text/html; charset=UTF-8',
            'From: ' . $from_name . ' <' . $from_email . '>',
        ];

        wp_mail($test_to, "Test Email – Easy Abandoned Cart", $message, $headers);
        echo "<div class='updated'><p>Test email sent to <strong>" . esc_html($test_to) . "</strong>.</p></div>";
    }

    // Load sender
    $sender_email = get_option('eac_sender_email', get_option('woocommerce_email_from_address'));
    $sender_name  = get_option('eac_sender_name', get_bloginfo('name'));

    echo '<form method="post">';
    for ($i = 1; $i <= 3; $i++) {
        $content = get_option("eac_email_template_$i") ?: "<h2>Hello {{first_name}},</h2><p>You left items in your cart.</p><p><a href='{{order_url}}'>Complete your order</a></p><p>Best regards, {{shop_name}} Team</p>";

        echo "<h2>Email #$i</h2>";
        echo '<p><small>Use: <code>{{first_name}}</code>, <code>{{shop_name}}</code>, <code>{{contact_email}}</code>, <code>{{order_url}}</code></small></p>';
        echo "<textarea name='eac_email_template_$i' rows='10' style='width:100%;'>" . esc_textarea($content) . "</textarea>";
        echo "<p><input type='submit' name='eac_save_template_$i' class='button-primary' value='Save Email #$i'></p>";
    }
    echo '</form>';

    // Sender settings
    echo '<hr><h2>📤 Sender Identity</h2>';
    echo '<form method="post">';
    echo '<p><label>Email address:</label><br>';
    echo '<input type="email" name="eac_sender_email" value="' . esc_attr($sender_email) . '" style="width:300px;"></p>';
    echo '<p><label>Sender name:</label><br>';
    echo '<input type="text" name="eac_sender_name" value="' . esc_attr($sender_name) . '" style="width:300px;"></p>';
    echo '<p><input type="submit" name="eac_save_sender" class="button-secondary" value="Save Sender Info"></p>';
    echo '</form>';

    // Test email
    echo '<hr><h2>✉️ Send Test Email</h2>';
    echo '<form method="post">';
    echo '<p><label>Email address to send test to:</label><br>';
    echo '<input type="email" name="eac_test_email" style="width:300px;" required></p>';
    echo '<p><label>Choose template (1–3):</label><br>';
    echo '<select name="eac_test_template">';
    for ($i = 1; $i <= 3; $i++) echo "<option value='$i'>Template #$i</option>";
    echo '</select></p>';
    echo '<p><input type="submit" name="eac_send_test_email" class="button" value="Send Test Email"></p>';
    echo '</form>';

    // Help section
    echo '<hr><h2>📘 How It Works</h2>';
    echo '<p>This plugin sends up to 3 follow-up emails to customers who abandon checkout (orders remain pending).</p>';
    echo '<ul><li>Email 1: sent after 1 hour</li><li>Email 2: sent after 24 hours</li><li>Email 3: sent after 3 days</li></ul>';
    echo '<p>Edit templates above and customize sender details. Use placeholders like <code>{{first_name}}</code>, <code>{{order_url}}</code>.</p>';
    echo '</div>';
}
